/*
 * Decompiled with CFR 0.152.
 */
package me.jddev0.ep.config;

import com.mojang.logging.LogUtils;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.nio.file.CopyOption;
import java.nio.file.Files;
import java.util.LinkedList;
import java.util.List;
import java.util.Optional;
import java.util.function.Function;
import me.jddev0.ep.config.ConfigValidationException;
import me.jddev0.ep.config.ConfigValue;
import org.slf4j.Logger;

public class Config {
    private static final Logger LOGGER = LogUtils.getLogger();
    private final List<ConfigValue<?>> configValues = new LinkedList();
    private final File configFile;
    private final String headerComment;
    private boolean loaded = false;

    public Config(File configFile, String headerComment) {
        this.configFile = configFile;
        this.headerComment = headerComment;
    }

    public boolean isRegistered(String key) {
        return this.configValues.stream().anyMatch(cv -> cv.getKey().equals(key));
    }

    public boolean isRegistered(ConfigValue<?> configValue) {
        return this.configValues.contains(configValue);
    }

    public <T> ConfigValue<T> register(ConfigValue<T> configValue) {
        String key = configValue.getKey();
        if (this.isRegistered(key)) {
            throw new RuntimeException("Config value conflict for key \"" + key + "\"");
        }
        this.configValues.add(configValue);
        return configValue;
    }

    public void unregister(ConfigValue<?> configValue) {
        String key = configValue.getKey();
        if (!this.isRegistered(key)) {
            throw new RuntimeException("Config value was not registered for key \"" + key + "\"");
        }
        this.configValues.removeIf(cv -> cv.getKey().equals(key));
    }

    public <T> ConfigValue<T> getConfigValue(String key) {
        Optional<ConfigValue> configValueOptional = this.configValues.stream().filter(cv -> cv.getKey().equals(key)).findAny();
        if (configValueOptional.isEmpty()) {
            throw new RuntimeException("Config value was not registered for key \"" + key + "\"");
        }
        return configValueOptional.get();
    }

    public static boolean isCharAtIndexEscaped(String str, int index) {
        if (str == null || str.length() <= index || index < 0) {
            return false;
        }
        for (int i = index - 1; i >= 0; --i) {
            if (str.charAt(i) == '\\') continue;
            return (index - i) % 2 == 0;
        }
        return index % 2 == 1;
    }

    public boolean isLoaded() {
        return this.loaded;
    }

    public void read() throws ConfigValidationException, IOException {
        this.loaded = true;
        if (!this.configFile.exists()) {
            this.configFile.getParentFile().mkdirs();
            this.configFile.createNewFile();
            this.writeDefault();
            this.configValues.forEach(ConfigValue::manuallyLoaded);
            return;
        }
        try (BufferedReader br = new BufferedReader(new FileReader(this.configFile));){
            String line;
            int lineNumber = 0;
            while ((line = br.readLine()) != null) {
                ++lineNumber;
                if (line.contains("#")) {
                    int commentStartIndex = -1;
                    while ((commentStartIndex = line.indexOf(35, commentStartIndex + 1)) > -1 && Config.isCharAtIndexEscaped(line, commentStartIndex)) {
                    }
                    if (commentStartIndex > -1) {
                        line = line.substring(0, commentStartIndex);
                    }
                }
                line = line.trim();
                StringBuilder lineBuilder = new StringBuilder(line);
                for (int unescapeIndex = line.length(); unescapeIndex > 0; --unescapeIndex) {
                    if (!Config.isCharAtIndexEscaped(line, unescapeIndex)) continue;
                    switch (lineBuilder.charAt(unescapeIndex)) {
                        case 'n': {
                            lineBuilder.replace(unescapeIndex, unescapeIndex + 1, "\n");
                            break;
                        }
                        case 't': {
                            lineBuilder.replace(unescapeIndex, unescapeIndex + 1, "\t");
                        }
                    }
                    lineBuilder.deleteCharAt(unescapeIndex - 1);
                    --unescapeIndex;
                }
                line = lineBuilder.toString();
                if (line.isEmpty()) continue;
                String[] tokens = line.split(" = ", 2);
                if (tokens.length != 2) {
                    String errorTxt = "Configuration file \"" + String.valueOf(this.configFile) + "\" has an invalid format at line " + lineNumber + ": \" = \" expected => Using default values.";
                    LOGGER.error(errorTxt);
                    throw new ConfigValidationException(errorTxt);
                }
                String key = tokens[0];
                if (!this.isRegistered(key)) {
                    LOGGER.warn("Configuration file \"" + String.valueOf(this.configFile) + "\" contains an invalid key at line " + lineNumber + ": \"" + key + "\" => Ignoring.");
                    continue;
                }
                ConfigValue configValue = this.getConfigValue(key);
                String value = tokens[1];
                try {
                    configValue.read(value);
                }
                catch (ConfigValidationException e) {
                    LOGGER.warn("Configuration file \"" + String.valueOf(this.configFile) + "\" contains an invalid value at line " + lineNumber + ": for \"" + key + "\" value \"" + value + "\" (" + e.getMessage() + ") => Using default value \"" + String.valueOf(configValue.getDefaultValue()) + "\".");
                }
            }
            if (this.configValues.stream().anyMatch(cv -> !cv.isLoaded())) {
                this.makeBackup();
                this.write();
                this.configValues.stream().filter(cv -> !cv.isLoaded()).forEach(ConfigValue::manuallyLoaded);
            }
        }
    }

    public void write() throws IOException {
        this.writeInternally(ConfigValue::write);
    }

    public void writeDefault() throws IOException {
        this.writeInternally(ConfigValue::writeDefault);
    }

    private void writeInternally(Function<ConfigValue<?>, String> configValueSerializer) throws IOException {
        if (!this.configFile.exists()) {
            this.configFile.getParentFile().mkdirs();
            this.configFile.createNewFile();
        }
        try (BufferedWriter bw = new BufferedWriter(new FileWriter(this.configFile));){
            bw.write("# === " + this.headerComment + " ===");
            bw.newLine();
            for (ConfigValue<?> configValue : this.configValues) {
                bw.newLine();
                String comment = configValue.getComment();
                if (comment != null) {
                    String[] commentLines;
                    for (String commentLine : commentLines = comment.split("\\n")) {
                        bw.write("# " + commentLine);
                        bw.newLine();
                    }
                    if (commentLines.length > 1) {
                        bw.write("#");
                        bw.newLine();
                    }
                }
                String defaultValue = configValue.writeDefault();
                defaultValue = defaultValue.replace("\\", "\\\\").replace("#", "\\#").replace("\n", "\\n").replace("\t", "\\t");
                bw.write("# [Default value]: " + defaultValue);
                bw.newLine();
                List<String> commentLines = configValue.getValidationCommentLines();
                for (String commentLine : commentLines) {
                    bw.write("# [Validation]: " + commentLine);
                    bw.newLine();
                }
                String rawValue = configValueSerializer.apply(configValue);
                rawValue = rawValue.replace("\\", "\\\\").replace("#", "\\#").replace("\n", "\\n").replace("\t", "\\t");
                bw.write(configValue.getKey() + " = " + rawValue);
                bw.newLine();
            }
            bw.flush();
        }
    }

    public void makeBackup() throws IOException {
        String fileName = this.configFile.getName();
        File backupConfigFile = new File(this.configFile.getParentFile(), fileName + ".5.bak");
        if (backupConfigFile.exists()) {
            backupConfigFile.delete();
        }
        for (int i = 5; i > 1; --i) {
            backupConfigFile = new File(this.configFile.getParentFile(), fileName + "." + i + ".bak");
            File newerBackupConfigFile = new File(this.configFile.getParentFile(), fileName + "." + (i - 1) + ".bak");
            if (!newerBackupConfigFile.exists()) continue;
            newerBackupConfigFile.renameTo(backupConfigFile);
        }
        backupConfigFile = new File(this.configFile.getParentFile(), fileName + ".1.bak");
        Files.copy(this.configFile.toPath(), backupConfigFile.toPath(), new CopyOption[0]);
    }
}

