#include "/lib/core.glsl"

#ifndef IS_IRIS
	#error This shader pack requires Iris: https://irisshaders.dev/
#endif

#include "/lib/config.glsl"

/*
	const float ambientOcclusionLevel = 0.8; // [0.0 0.1 0.2 0.3 0.4 0.5 0.6 0.7 0.8 0.9 1.0]
*/

#ifdef SKIP_CLEAR
#endif

#if defined END || (defined SKIP_CLEAR && !defined NETHER)
	/*
		const bool colortex0Clear = false;
	*/
#endif

#if COLOR_FORMAT == 0
	/*
		const int colortex0Format = RGB8;
	*/
#elif COLOR_FORMAT == 1
	/*
		const int colortex0Format = R11F_G11F_B10F;
	*/
#elif COLOR_FORMAT == 2
	/*
		const int colortex0Format = RGB16F;
	*/
#elif COLOR_FORMAT == 3
	/*
		const int colortex0Format = RGB32F;
	*/
#endif

/* RENDERTARGETS: 0 */
layout(location = 0) out vec3 outColor0;

uniform sampler2D colortex0;

in VertexData {
	vec2 coord;
} vsh;

#if defined FXAA || BLOOM
	uniform float viewHeight, viewWidth;
#endif

#if BLOOM
	uniform sampler2D noisetex;
#endif

#ifdef AUTOMATIC_EXPOSURE
	uniform ivec2 eyeBrightnessSmooth;

	uniform int worldTime;
#endif

vec3 tonemap(in vec3 color) {
	#if !TONEMAP
		return color;
	#elif TONEMAP == 1 // Uchimura
		vec3 w0 = 1.0 - smoothstep(0.0, 0.22, color);
		vec3 w2 = step(0.532, color);

		return 0.22 * pow(color / 0.22, vec3(1.33)) * w0 + (0.22 + color - 0.22) * (1.0 - w0 - w2) + (1.0 - 0.468 * exp((color - 0.532) / -0.468)) * w2;
	#elif TONEMAP == 2 // Reinhard-Jodie
		color = pow(color, vec3(TONEMAP_CURVE));

		immut vec3 tv = color / (1.0 + color);

		return pow(mix(color / (dot(color, vec3(0.2125, 0.7154, 0.0721)) + 1.0), tv, tv), vec3(1.0 / TONEMAP_CURVE));
	#elif TONEMAP == 3 // ACES
		return color * (2.51 * color + 0.03) / (color * (2.43 * color + 0.59) + 0.14);
	#elif TONEMAP == 4 // Lottes
		const float a = (pow(8.0, 1.5632) - pow(0.18, 1.5632)) * 0.267;
		const float b = (pow(8.0, 1.6) * 0.267 - pow(0.18, 1.6)) / a;
		const float c = (pow(8.0, 1.5632) * pow(0.18, 1.6) - pow(8.0, 1.6) * pow(0.18, 1.5632) * 0.267) / a;

		return pow(color, vec3(1.6)) / (pow(color, vec3(1.5632)) * b + c);
	#elif TONEMAP == 5 // Unreal
		return pow(color / (color + 0.155) * 1.019, vec3(2.2));
	#elif TONEMAP == 6 // Filmic
		color = max(vec3(0.0), color - 0.004);

		return pow(color * (6.2 * color + 0.5) / (color * (6.2 * color + 1.7) + 0.06), vec3(2.2));
	#elif TONEMAP == 7 // Burgess HDR
		color = max(vec3(0.0), mix(color * 0.6, color * 1.2, color) - 0.004);

		return color * (6.2 * color + 0.05) / (color * (6.2 * color + 2.3) + 0.06);
	#elif TONEMAP == 8 // Uncharted 2
		immut vec3 col_2 = color * 2.0;

		return ((col_2 * (0.15 * col_2 + 0.05) + 0.004) / (col_2 * (0.15 * col_2 + 0.5) + 0.06) - 0.2 / 3.0) / (19.38 / 24.476 - 0.2 / 3.0);
	#endif
}

#ifdef FXAA
	float luminance(in vec3 color) {
		return dot(color, vec3(0.299, 0.587, 0.114));
	}
#endif

void main() {
	vec3 color = texture(colortex0, vsh.coord).rgb;

	#ifdef FXAA
		// FXAA 3.11 from:
		// http://blog.simonrodriguez.fr/articles/30-07-2016_implementing_fxaa.html
		// https://github.com/kosua20/Rendu/blob/master/resources/common/shaders/screens/fxaa.frag

		immut float luma_center = luminance(color);
		immut float luma_down = luminance(textureOffset(colortex0, vsh.coord, ivec2(0, -1)).rgb);
		immut float luma_up = luminance(textureOffset(colortex0, vsh.coord, ivec2(0, 1)).rgb);
		immut float luma_left = luminance(textureOffset(colortex0, vsh.coord, ivec2(-1, 0)).rgb);
		immut float luma_right = luminance(textureOffset(colortex0, vsh.coord, ivec2(1, 0)).rgb);

		immut float luma_min = min(luma_center, min(min(luma_down, luma_up), min(luma_left, luma_right)));
		immut float luma_max = max(luma_center, max(max(luma_down, luma_up), max(luma_left, luma_right)));

		immut float luma_range = luma_max - luma_min;

		if (luma_range > max(0.03125, luma_max * 0.125)) {
			const float quality[12] = float[12] (
				1.0,
				1.0,
				1.0,
				1.0,
				1.0,
				1.5,
				2.0,
				2.0,
				2.0,
				2.0,
				4.0,
				8.0
			);

			immut float luma_down_left = luminance(textureOffset(colortex0, vsh.coord, ivec2(-1, -1)).rgb);
			immut float luma_up_right = luminance(textureOffset(colortex0, vsh.coord, ivec2(1, 1)).rgb);
			immut float luma_up_left = luminance(textureOffset(colortex0, vsh.coord, ivec2(-1, 1)).rgb);
			immut float luma_down_right = luminance(textureOffset(colortex0, vsh.coord, ivec2(1, -1)).rgb);

			immut float luma_down_up = luma_down + luma_up;
			immut float luma_left_right = luma_left + luma_right;

			immut float luma_left_corners = luma_down_left + luma_up_left;
			immut float luma_down_corners = luma_down_left + luma_down_right;
			immut float luma_right_corners = luma_down_right + luma_up_right;
			immut float luma_up_corners = luma_up_right + luma_up_left;

			immut float edge_horizontal =
				abs(-2.0 * luma_left + luma_left_corners) +
				abs(-2.0 * luma_center + luma_down_up) * 2.0 +
				abs(-2.0 * luma_right + luma_right_corners);
			immut float edge_vertical =
				abs(-2.0 * luma_up + luma_up_corners) +
				abs(-2.0 * luma_center + luma_left_right) * 2.0 +
				abs(-2.0 * luma_down + luma_down_corners);

			immut bool is_horizontal = (edge_horizontal >= edge_vertical);

			immut float luma_1 = is_horizontal ? luma_down : luma_left;
			immut float luma_2 = is_horizontal ? luma_up : luma_right;
			immut float gradient_1 = luma_1 - luma_center;
			immut float gradient_2 = luma_2 - luma_center;

			immut float gradient_scaled = 0.25 * max(abs(gradient_1), abs(gradient_2));

			immut vec2 view = 1.0 / vec2(viewWidth, viewHeight);

			float step_length = is_horizontal ? view.y : view.x;

			float luma_local_average = 0.0;

			if (abs(gradient_1) >= abs(gradient_2)) {
				step_length = - step_length;
				luma_local_average = 0.5 * (luma_1 + luma_center);
			} else {
				luma_local_average = 0.5 * (luma_2 + luma_center);
			}

			vec2 current_uv = vsh.coord;

			if (is_horizontal) {
				current_uv.y += step_length * 0.5;
			} else {
				current_uv.x += step_length * 0.5;
			}

			immut vec2 offset = is_horizontal ? vec2(view.x, 0.0) : vec2(0.0, view.y);

			vec2 uv1 = current_uv - offset;
			vec2 uv2 = current_uv + offset;

			float luma_end_1 = luminance(texture(colortex0, uv1).rgb) - luma_local_average;
			float luma_end_2 = luminance(texture(colortex0, uv2).rgb) - luma_local_average;

			bool reached_1 = abs(luma_end_1) >= gradient_scaled;
			bool reached_2 = abs(luma_end_2) >= gradient_scaled;

			bool reached_both = reached_1 && reached_2;

			if (!reached_1) uv1 -= offset;
			if (!reached_2) uv2 += offset;

			if (!reached_both) {
				for(uint i = 2; i < 12; ++i) {
					if (!reached_1) luma_end_1 = luminance(texture(colortex0, uv1).rgb) - luma_local_average;
					if (!reached_2) luma_end_2 = luminance(texture(colortex0, uv2).rgb) - luma_local_average;

					reached_1 = abs(luma_end_1) >= gradient_scaled;
					reached_2 = abs(luma_end_2) >= gradient_scaled;
					reached_both = reached_1 && reached_2;

					if (!reached_1) uv1 -= offset * quality[i];
					if (!reached_2) uv2 += offset * quality[i];

					if (reached_both) break;
				}
			}

			immut float distance_1 = is_horizontal ? (vsh.coord.x - uv1.x) : (vsh.coord.y - uv1.y);
			immut float distance_2 = is_horizontal ? (uv2.x - vsh.coord.x) : (uv2.y - vsh.coord.y);

			immut bool correct_variation = ((distance_1 < distance_2 ? luma_end_1 : luma_end_2) < 0.0) != luma_center < luma_local_average;

			float final_offset = correct_variation ? -min(distance_1, distance_2) / (distance_1 + distance_2) + 0.5 : 0.0;

			immut float luma_average = (1.0 / 12.0) * (2.0 * (luma_down_up + luma_left_right) + luma_left_corners + luma_right_corners);
			immut float sub_pixel_offset_1 = clamp(abs(luma_average - luma_center) / luma_range, 0.0, 1.0);
			immut float sub_pixel_offset_2 = (-2.0 * sub_pixel_offset_1 + 3.0) * sub_pixel_offset_1 * sub_pixel_offset_1;

			final_offset = max(final_offset, sub_pixel_offset_2 * sub_pixel_offset_2 * 0.75);

			vec2 final_uv = vsh.coord;

			if (is_horizontal) {
				final_uv.y += final_offset * step_length;
			} else {
				final_uv.x += final_offset * step_length;
			}

			color = texture(colortex0, final_uv).rgb;
		}
	#endif

	#if BLOOM
		const vec2[9] offsets = vec2[9](
			vec2(0.0, 0.0),
			vec2(0.0, 1.0),
			vec2(0.7, 0.7),
			vec2(1.0, 0.0),
			vec2(0.7,-0.7),
			vec2(0.0,-1.0),
			vec2(-0.7,-0.7),
			vec2(-1.0, 0.0),
			vec2(-0.7, 0.7)
		);

		immut float noise = texture(noisetex, gl_FragCoord.xy / noiseTextureResolution).r;

		immut float noise_cos = cos(noise);
		immut float noise_sin = sin(noise);

		immut mat2 rotation = mat2(
			noise_cos, -noise_sin,
			noise_sin, noise_cos
		) * 0.0025;

		vec3 sum = vec3(0.0);

		for(uint i = 0; i < offsets.length(); ++i) {
			sum += texture(colortex0, BLOOM_RADIUS * vec2(offsets[i]) * rotation + vsh.coord).rgb;
		}

		sum /= offsets.length();

		sum *= sum * sum * sum;

		// float brightness = 1.0 + max(color.r, max(color.g, color.b));
		// sum /= brightness;

		color += 0.1 * BLOOM * sum;
	#endif

	color = pow(color, vec3(2.4));

	#ifdef AUTOMATIC_EXPOSURE
		immut float torch = eyeBrightnessSmooth.s / 30.0;
		immut float sky = eyeBrightnessSmooth.t / 240.0 + 0.5 - distance(worldTime, 6000.0) / 12000.0;

		color *= max(1.0, 3.0 - mix(torch, 2.0, clamp(sky, 0.0, 1.0)));
	#endif

	#if HDR_OVEREXPOSED != 100 || HDR_UNDEREXPOSED != 100
		color = mix(color * HDR_UNDEREXPOSED * 0.01, color * HDR_OVEREXPOSED * 0.01, color);
	#endif

	#if RED != 100 || GREEN != 100 || BLUE != 100
		color *= vec3(RED, GREEN, BLUE) * 0.01;
	#endif

	#if SATURATION != 100
		color = mix(dot(color, vec3(0.2126, 0.7152, 0.0722)).rrr, color, SATURATION * 0.01);
	#endif

	outColor0 = pow(tonemap(color), vec3(1.0 / GAMMA));
}